import struct
import logging
import logging.config

import warnings
warnings.simplefilter("ignore", category=DeprecationWarning)

from twisted.internet import ssl, reactor
from twisted.internet.protocol import Factory, Protocol

import valueslib


log = None


class NotEnoughDataError(Exception):
    pass


def makecmd(command, data):
    return struct.pack(">H", len(data)) + "\x00\x00" + chr(command) + data

def parsecmd(data):
    if len(data) < 2:
        raise NotEnoughDataError('need more data')

    cmdlen, = struct.unpack('>H', data[0:2])
    if len(data) < cmdlen + 5:
        raise NotEnoughDataError('need more data')

    b23 = data[2:4]
    cmd = ord(data[4])
    seqnum = data[5:9]
    segment = data[9:cmdlen + 5]

    return cmd, b23, seqnum, segment, data[cmdlen + 5:]


class UServerBase(Protocol):
    buffer = ''

    def __init__(self):
        self.log = logging.getLogger(self.__class__.__name__)

    def dataReceived(self, data):
        self.buffer += data

        self.log.debug('data received: %r', data.encode('hex'))

        m = getattr(self, 'handle_raw', None)
        if m:
            while self.buffer:
                self.buffer = m(self.buffer)
            return

        while 1:
            try:
                cmd, b23, seqnum, segment, self.buffer = parsecmd(self.buffer)
            except NotEnoughDataError, e:
                return

            m = getattr(self, 'handle_cmd_%02x' % (cmd,), None)
            if not m:
                self.handle_unknown(cmd, b23, seqnum, segment)
            else:
                msg = m(cmd, b23, seqnum, segment)

                if msg is not None:
                    self.log.debug('sending data: %r', msg.encode('hex'))
                    self.transport.write(msg)

    def handle_unknown(self, cmd, b23, seqnum, segment):
        raise RuntimeError('Unimplemented command type: 0x%02x' % (cmd))


class UServer1(UServerBase):
    def connectionMade(self):
        self.transport.write(makecmd(0x30, ''))

    def handle_cmd_0e(self, cmd, b23, seqnum, segment):
        if segment not in ('\x00\x00\x00\x04\x00\x00\x00\x06',
                           '\x00\x00\x00\x04\x00\x00\x00\x07'):
            raise RuntimeError('Unknown command: 0x%02x : %r', cmd,
                               segment.encode('hex'))
        return makecmd(0x0f, seqnum +
                       ('\x00\x00\x00\x00\xff\xff\xff\xff\xff'
                        '\xff\xff\xff\x01\x00\x00\x7f\x32\xcd'))

    def handle_cmd_12(self, cmd, b23, seqnum, segment):
        if segment != '\x00\x00\x00\x04':
            raise RuntimeError('Unknown command: 0x%02x : %r', cmd,
                               segment.encode('hex'))
        return makecmd(0x13, seqnum + '\x01\x00\x17xxxxxxxxxxxxxxx' +
                       'xxxxxxxx\x00\x00')

    def handle_cmd_16(self, cmd, b23, seqnum, segment):
        if segment != ('\x00\x00\x00\x00\x00\x00\x00\x00\x00'
                       '\x00\x00\x03\x00\x00\x00\x00\x00\x00'):
            raise RuntimeError('Unknown command: 0x%02x : %r', cmd,
                               segment.encode('hex'))
        return makecmd(0x17, seqnum +
                       ('\xcc\xcc\xcc\xcc\x00\x03GET\x00\x15?prefix=xxxxxxx'
                        'xxxxx/\x00(ubisoft-orbit-savegames.s3.amazonaws.xx'
                        'x\x00\xa6Host: ubisoft-orbit-savegames.s3.amazonaw'
                        's.xxx\nDate: xxxxxxxxxxxxxxxxxxxxxxxxx GMT\nConten'
                        't-MD5: \nAuthorization: AWS xxxxxxxxxxxxxxxxxxxx:x'
                        'xxxxxxxxxxxxxxxxxxxxxxxxxxx\n'))

    def handle_cmd_18(self, cmd, b23, seqnum, segment):
        if segment != '\x00\x00\x00\x04\x00\x00\x00\x00':
            raise RuntimeError('Unknown command: 0x%02x : %r', cmd,
                               segment.encode('hex'))
        return makecmd(0x19, seqnum + '\x00\x00\x00')

    def handle_cmd_1a(self, cmd, b23, seqnum, segment):
        if segment != '':
            raise RuntimeError('Unknown command: 0x%02x : %r', cmd,
                               segment.encode('hex'))
        return makecmd(0x1b, seqnum + '\x00\x02\x4e\x4f')

    def handle_cmd_31(self, cmd, b23, seqnum, segment):
        return makecmd(0x01, seqnum + '\x00$xxxxxxxxxxxxxxxxxxxxxx' +
                       'xxxxxxxxxxxxxx\x00\x00\x00\x00')


class UServer2(UServerBase):
    def handle_raw(self, data):
        if data[0:5] == '\x00\x08\x00\x00\x00':
            msg = '\x00\x00\x00\x00\x02'
        elif data == '\x00\x00\x00\x00\x03':
            msg = '000b000001003390ffffffffffffffff'.decode('hex')
        else:
            raise RuntimeError('Unknown data! %r', data.encode('hex'))

        self.log.debug('sending data: %r', msg.encode('hex'))
        self.transport.write(msg)

        return ''


class UServer3(UServerBase):
    db = None
    activated = None

    def connectionMade(self):
        self.db = valueslib.db()
        self.activated = {}

    def handle_raw(self, data):
        msg = None

        if data[0:5] == '\x00\x08\x00\x00\x00':
            msg = '\x00\x00\x00\x00\x01'
        elif data[0:5] == '\x00\x14\x00\x00\x03':
            base = data[5:9]
            self.activated[base] = True
            msg = None
        else:
            key = data.encode('hex')
            if key in self.db.pairs:
                value = self.db.pairs[key].decode('hex')
                msg = ''
                for i in xrange(0, len(value), 23):
                    subvalue = value[i:i+23]
                    if subvalue[9:13] in self.activated:
                        msg += subvalue
                        self.log.debug('returning challenge response %s',
                                       subvalue.encode('hex'))
                    else:
                        self.log.warning('Challenge with no base before!')
            else:
                self.log.warning('Unknown challenge: %s', key)

        if msg is not None:
            self.log.debug('sending data: %r', msg.encode('hex'))
            self.transport.write(msg)

        return ''


def web_serve(port, root_path):
    from twisted.web import server, static

    root = static.File(root_path)
    site = server.Site(root)

    res = reactor.listenTCP(port, site)
    log.info('Started http handler on port %d', port)

    return res


def set_logging():
    global log
    logging.config.fileConfig('tserver_logging.conf')

    log = logging.getLogger('tserver')

    from twisted.python import log as twlog
    PLO = twlog.PythonLoggingObserver
    class DebugLevelObserver(PLO):
        def emit(self, eventDict):
            if 'logLevel' not in eventDict and not eventDict['isError']:
                eventDict['logLevel'] = logging.DEBUG
            return PLO.emit(self, eventDict)
    observer = DebugLevelObserver('tw')
    twlog.startLoggingWithObserver(observer.emit, setStdout=0)


def main():
    def listen_with_protocol(port, proto_class, key_file, cert_file):
        factory = Factory()
        factory.protocol = proto_class
        ctx = ssl.DefaultOpenSSLContextFactory(key_file, cert_file)
        res = reactor.listenSSL(port, factory, ctx)

        log.info('Started %r handler on port %d', proto_class.__name__,
                 port)
        return res

    key_file = 'key1.pem'
    cert_file = 'cert1.pem'
    www_path = './www'

    listen_with_protocol(13000, UServer1, key_file, cert_file)
    listen_with_protocol(13005, UServer2, key_file, cert_file)
    listen_with_protocol(13200, UServer3, key_file, cert_file)

    try:
        web_serve(80, www_path)
    except Exception, e:
        log.warning('Failed to start web server: %r', e)

    reactor.run()

    log.info('Reactor stopped, bye.')



if __name__ == '__main__':
    set_logging()

    main()
